/*****************************************************************************/

/*
 *
 *   Copyright (c) 2002, Smart Link Ltd.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *       1. Redistributions of source code must retain the above copyright
 *          notice, this list of conditions and the following disclaimer.
 *       2. Redistributions in binary form must reproduce the above
 *          copyright notice, this list of conditions and the following
 *          disclaimer in the documentation and/or other materials provided
 *          with the distribution.
 *       3. Neither the name of the Smart Link Ltd. nor the names of its
 *          contributors may be used to endorse or promote products derived
 *          from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *
 *	mdm_params.c  --  Smart Link Soft Modem run time params helper.
 *
 *	Author: Sasha K (sashak@smlink.com)
 *
 */

/*****************************************************************************/

#define __KERNEL_SYSCALLS__
#include <linux/kernel.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/unistd.h>
#include <linux/vmalloc.h>
#include <linux/errno.h>
#include <asm/uaccess.h>

#include <linux_compat.h>

static int errno;

static const char *param_file = "/var/lib/slmdm.data";

int load_runtime_params(char *params, int len)
{
	mm_segment_t fs;
	int ret = -1;
	int fd;

	fs = get_fs();
	set_fs(get_ds());

	fd = open(param_file, 0, 0);
	if(fd < 0) {
		if(errno != ENOENT) {
			printk(KERN_WARNING "slmdm: cannot open file `%s': errno %d\n",
		       		param_file,errno);
		}
		goto error;
        }

	ret = read(fd,params,len);
	if (ret < 0) {
                printk(KERN_ERR "slmdm: cannot read file `%s': errno %d\n",
		       param_file,errno);
        }

	close(fd);
 error:
	set_fs(fs);
	return ret;
}

int save_runtime_params(char *params, int len)
{
	mm_segment_t fs;
	int ret = -1;
	int fd;

	if(!current->files)
		return -1;

	fs = get_fs();
	set_fs(get_ds());

	fd = open(param_file,0101/* O_CREAT|O_WRONLY */ ,0600);
	if(fd < 0) {
		printk(KERN_WARNING "slmdm: cannot open file `%s': errno %d\n",
		       param_file,errno);
		goto error;
        }

	ret = write(fd,params,len);
	if (ret < 0) {
                printk(KERN_ERR "slmdm: cannot write file `%s': errno %d\n",
		       param_file,errno);
        }

	close(fd);
 error:
	set_fs(fs);
	return ret;
}


/*
 *
 *  Country parameters set related stuff
 *
 */

#define COUNTRY_NAME_LENGTH 24
#define COUNTRY_INDEX_SIZE  512

struct country_set {
	unsigned id;
	char name[COUNTRY_NAME_LENGTH];
	unsigned char setting[194];
};

struct country_index {
	unsigned id;
	char name[COUNTRY_NAME_LENGTH];
	unsigned offset;
};



/* default country set definitions */
static struct country_set this_country_set =
{ 0xb5, "USA", {
        0x24,0x40,0x1,0x6,0x32,0x96,0x2b,0x0,0xff,0x41,0x1,0xff,
        0xa,0x50,0xf,0x23,0x12,0x3c,0x4,0x12,0x3c,0x0,0xa,0x0,
        0x5,0x1,0x0,0x1,0x78,0x0,0x0,0x1,0x1,0x0,0x1,0x0,
        0x5,0x32,0x2,0x41,0x1,0xff,0x2,0x41,0x1,0xff,0xa,0x0,
        0x3,0x0,0x1,0xff,0x1,0x19,0x1,0x2,0x0,0x0,0x5,0x2b,
        0x0,0xff,0xa,0x2b,0x32,0x2b,0x0,0x2b,0x0,0x0,0x0,0x0,
        0x0,0x1,0x4,0x1e,0x19,0x32,0xfa,0xfa,0x1,0x1,0x0,0x6,
        0x0,0x50,0x2,0x2,0x0,0x13,0x0,0x64,0x3c,0x5,0x0,0x0,
        0x0,0x35,0xa,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,
        0x0,0x0,0x0,0x0,0x6,0x42,0x0,0xa1,0x24,0x4,0x81,0x0,
        0x0,0x0,0x64,0x0,0xf,0x3c,0x16,0x50,0x0,0x1,0xb5,0x0,
        0x78,0x55,0x0,0x0,0x0,0x0,0x0,0x0,0x7,0x0,0x0,0x0,
        0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,
        0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,
        0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,
        0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,
        0x0, } };

/* small strcmp */
extern inline int case_insens_strcmp(char *s1, char *s2)
{
	for (; tolower(*s1) == tolower(*s2); s1++, s2++) {
		if (*s1 == '\0')
			break;
	}
	return *s1 - *s2;
}
#ifdef strcmp			/* sometimes macro */
#undef strcmp
#endif
#define strcmp(s1,s2) case_insens_strcmp(s1,s2)


struct country_set *init_country_set(char *country_file, int id, char *name)
{
	mm_segment_t fs;
	struct country_set *country_set = &this_country_set;
	struct country_index *index;
	int fd;

	if ((id < 0 && ( !name || !*name)) ||    /* force default */
	    id == country_set->id ||
	    (id < 0 && !strcmp(country_set->name,name)))
		goto ok;

	fs = get_fs();
	set_fs(get_ds());

	if (!country_file || !*country_file ||
	    (fd = open(country_file, 0, 0)) < 0) {
		printk(KERN_ERR "slmdm: cannot open country file `%s'.\n",
		       country_file?country_file:"");
		goto error;
        }

	index = (struct country_index *)vmalloc(COUNTRY_INDEX_SIZE*sizeof(*index));
	if (!index) {
		printk(KERN_ERR "slmdm: out of mem loading `%s'.\n",
		       country_file);
		goto error1;
	}
		
        if ( read(fd, (void *)index, COUNTRY_INDEX_SIZE*sizeof(*index)) < 0 ) {
                printk(KERN_ERR "slmdm: cannot read file `%s'.\n", country_file);
		goto error2;
        }

	if ( id >= 0 ) { /* search by id */
		if (id >= COUNTRY_INDEX_SIZE || !*index[id].name) {
			printk(KERN_ERR "slmdm: unknown country code %x.\n", id);
			goto error2;
		}
	} else {   /* search by name */
		for ( id = 0 ; id < COUNTRY_INDEX_SIZE ; id++ ) {
			if (!strcmp(index[id].name,name)) {
				break;
			}
		}
		if (id >= COUNTRY_INDEX_SIZE) {
			printk(KERN_ERR "slmdm: unknown country name `%s'.\n", name);
			goto error2;
		}
	}

	country_set->id = id;
	memcpy(country_set->name, index[id].name, sizeof(country_set->name));

	if (lseek(fd, COUNTRY_INDEX_SIZE*sizeof(*index) + index[id].offset, 0) < 0) {
		printk(KERN_ERR "slmdm: cannot seek `%s'.\n", country_file);
		goto error2;
	}

#ifdef DEBUG_COUNTRY
	if (read(fd, country_set, sizeof(*country_set)) < 0) {
#else
	if (read(fd, country_set->setting, sizeof(country_set->setting)) < 0) {
#endif
		printk(KERN_ERR "slmdm: cannot read set from `%s'.\n", country_file);
		goto error2;
	}

#ifdef DEBUG_COUNTRY
	if (country_set->id != id) {
		printk(KERN_ERR "slmdm: bad country id (%x != %x).\n",
		       country_set->id, id);
		goto error2;
	}
#endif
 error2:
	vfree(index);
 error1:
	close(fd);
 error:
	set_fs(fs);
 ok:
	printk(KERN_INFO "slmdm: country set is 0x%x (%s).\n",
	       country_set->id, country_set->name);
	return country_set;
}

