/*****************************************************************************/

/*
 *
 *   Copyright (c) 2002, Smart Link Ltd.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *       1. Redistributions of source code must retain the above copyright
 *          notice, this list of conditions and the following disclaimer.
 *       2. Redistributions in binary form must reproduce the above
 *          copyright notice, this list of conditions and the following
 *          disclaimer in the documentation and/or other materials provided
 *          with the distribution.
 *       3. Neither the name of the Smart Link Ltd. nor the names of its
 *          contributors may be used to endorse or promote products derived
 *          from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *
 *	usb_st7554.c  --  ST7554 USB Smart Link Soft Modem driver
 *
 *	Author: SashaK (sashak@smlink.com)
 *
 *
 */

/*****************************************************************************/

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/list.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/tqueue.h>
#include <asm/semaphore.h>

#include <linux/usb.h>

#include <linux_compat.h>

/* #include <modem.h> */
#include <modem_driver.h>

#define USB_INFO(fmt...) printk(KERN_INFO fmt)
#define USB_ERR(fmt...)  printk(KERN_ERR fmt)

#if MODEM_DEBUG
#include <modem_debug.h>
#define printk mprintf
static int debug = 0;
MODULE_PARM(debug,"i");
MODULE_PARM_DESC(debug,"Debug level: 0-3 (default=0)");
#define USB_DBG(fmt...) { if(debug) printk(KERN_DEBUG fmt); }
//#define DEBUG_URB_PRINT 1
#define USB_DBG_URB(fmt...) // USB_DBG(fmt)
#else  /* not MODEM_DEBUG */
#define USB_DBG(fmt,arg...)
#define USB_DBG_URB(fmt...)
#endif /* not MODEM_DEBUG */


/* st7554 IDs */
#define ST7554_ID_VENDOR   0x0483
#define ST7554_ID_PRODUCT  0x7554

/* st7554 interfaces */
#define ST7554_IFACE_CTRL_ALTSETTING         0
#define ST7554_IFACE_MODEM_ALTSETTING        1

/* comm class defined requests */
#define USB_COMM_CLASS_AUX_LINE_STATE 0x0a
#define USB_COMM_CLASS_HOOK_STATE 0x0b
#define USB_COMM_CLASS_PULSE_STATE 0x0c

/* st7554 vendor defined requests */
#define ST7554_SSI1_CONTROL    0x1
#define ST7554_SSI2_CONTROL    0x2
#define ST7554_FIFO_MASK       0x3
#define ST7554_SSI1_COUNTER    0x4
#define ST7554_SSI2_COUNTER    0x6
#define ST7554_GPIO_DIR        0x8
#define ST7554_GPIO_OUT        0xA
#define ST7554_GPIO_MASK       0xC
#define ST7554_GPIO_INV        0xE
#define ST7554_GPIO_STATUS     0x15
#define ST7554_SSI1_CWORD      0x17
#define ST7554_FIFO_CONTROL1   0x19
#define ST7554_FIFO_CONTROL2   0x1a

/* requests' values */
/* SSI control */
#define SSI1_POWERDOWN       0x0000 
#define SSI1_POWERUP         0x0001
/* FIFO mask */
#define SSI1_UNDERRUN        0x0040
#define SSI1_OVERRUN         0x0020
/* GPIO mask */
#define GPIO_RING1           0x0001
#define GPIO_HANDSET         0x0004
#define GPIO_HOOK1           0x0008
#define GPIO_CID1            0x0020
#define GPIO_LED_CARRIER     0x0040
#define GPIO_LED_HOOK        0x0080
#define GPIO_RFC             0x0100
#define GPIO_DISHS           0x0200
#define GPIO_DP              0x0400  /* used for Off Hook */
#define GPIO_BUZEN           0x0800
#define GPIO_DC              0x1000
#define GPIO_LED_RW          0x2000
#define GPIO_MONOLITHINC     0x4000
/* FIFO control */
#define BYTE_ORDER_LE        0x8000

/* gpio hook off bit mask */
#define GPIO_HOOK1_OFF   (GPIO_HOOK1)
//#define GPIO_HOOK1_OFF   (GPIO_HOOK1|GPIO_DP|GPIO_DC)

/* fifo size */
#define ST7554_FIFO_SIZE 128

/* urb size */
#define DESCFRAMES 5
/* skip on start */
#define SKIPFRAMES 3

/* control message timeout */
#define CONTROL_MSG_TMO HZ


/* data type definitions */

struct st7554_modem_state {
	const char *name;
	struct usb_device    *usbdev; 
	struct usb_interface *iface ;
	struct list_head     list;
	spinlock_t lock;

	unsigned int ctrl_ep; /* control endpoint */
	u16 gpio_out;         /* gpio_out register map */
	u32 intr_status;      /* interrupt status */
	urb_t intr_urb;       /* interrupt urb */

	struct modem *modem;  /* modem high level instance */

	struct tq_struct ring_bh;  /* ring BH  */
	struct tq_struct modem_bh; /* modem BH */

	unsigned int hook;         /* hook state */
	unsigned int format;       /* sample format */
	unsigned int srate;        /* sample rate */
	unsigned int fragsize;     /* fragsize in bytes */
	unsigned int mapped;       /* mapped modem channels */

	int delay;                 /* i/o delay */
	int disbalance;            /* output disbalance */

	struct usb_modem_channel { /* modem in/out channels */
		unsigned int maxsz;  /* max packet size */
		unsigned int pipe;   /* usb data pipe */
		struct dmabuf {	     /* "dma" ring buffer */
			int          count;    /* byte count */
			unsigned int head;     /* head(write) pointer */
			unsigned int tail;     /* tail(read) pointer */
			unsigned int size;     /* buffer size */
			unsigned char *buf;    /* data buffer */
			unsigned int error;    /* over/underrun */
		} dma;
		struct urb *urb[2];  /* isoc urb */
	} mo, mi;

	struct codec {             /* installed codec */
		const char *name;
		int (*set_srate )(struct st7554_modem_state *,unsigned);
		int (*set_format)(struct st7554_modem_state *,unsigned);
	} codec;

	/* init proc */
	int (*init)(struct st7554_modem_state *);
	int (*release)(struct st7554_modem_state *);
	/* register access proc */
	int (*get_reg)(struct st7554_modem_state *s, u8 reg, u16 *value);
	int (*set_reg)(struct st7554_modem_state *s, u8 reg, u16  value);
};


/*
 * linked list of all st7554 usb devices...
 */

static struct list_head st7554_devs = LIST_HEAD_INIT(st7554_devs);

static DECLARE_MUTEX(open_sem);


/* --------------------------------------------------------------------- */
/* dma init */

static int dma_init (struct dmabuf *db)
{
	db->buf = (unsigned char *)__get_free_pages(GFP_KERNEL, 1);
	if (!db->buf)
		return -ENOMEM;
	db->head = db->tail = db->count = 0;
	db->size = 1UL<<(PAGE_SHIFT + 1) ;
	return 0;
}


static void dma_free(struct dmabuf *db)
{
	int size = db->size;
	db->head = db->tail = db->count = db->size = 0;
	free_pages((unsigned long)db->buf, get_order(size));
	db->buf = NULL;
}


static int dmabuf_copyin(struct dmabuf *db, void *buffer, unsigned int size)
{
        int ret = 0, cnt;
        while (size) {
		cnt = db->size - db->head;
		if (cnt > size )
			cnt = size;
		if (cnt > db->size - db->count)
			cnt = db->size - db->count;
		if (cnt <= 0) {      /* overflow */
			db->error++;
			USB_ERR("dmabuf_copyin: overrun: ret %d.\n", ret);
			return ret;
		}
		memcpy(db->buf + db->head, buffer, cnt);
		
                buffer += cnt;
		db->count += cnt;
                db->head += cnt ;
                if (db->head >= db->size)
                        db->head = 0;
		size -= cnt;
		ret += cnt;
        }
	return ret;
}


static int dmabuf_copyout(struct dmabuf *db, void *buffer, unsigned int size)
{
	int ret = 0, cnt;

	while (size) {
		cnt = db->size - db->tail;
		if ( cnt > size )
			cnt = size;
		if ( cnt > db->count )
			cnt = db->count;
		if ( cnt <= 0 ) {  /* underflow */
			//db->error++;
			//USB_ERR("dmabuf_copyout: underrun: ret %d (skip %d).\n", ret, size);
			return ret;
		}
		memcpy(buffer, db->buf + db->tail, cnt);

		(char *)buffer += cnt;
		db->count -= cnt;
		db->tail += cnt;
                if (db->tail >= db->size)
                        db->tail = 0;
                size -= cnt;
		ret += cnt;
        }
	return ret;
}


/* --------------------------------------------------------------------- */

#define NUM_OF_URBS(ch) (sizeof((ch)->urb)/sizeof((ch)->urb[0]))
#define MO_URB_NO(s,u) ((u) == (s)->mo.urb[1])
#define MI_URB_NO(s,u) ((u) == (s)->mi.urb[1])

#define BYTES_IN_FRAMES(s,n) ((((s)->srate*(n))/1000)<<(MFMT_BYTESSHIFT((s)->format)))

#define FILL_URB(state,ch,u) { \
	(u)->dev = (state)->usbdev;  \
	(u)->context  = (state);          \
	(u)->number_of_packets = DESCFRAMES;  \
        (u)->status         = 0;              \
	(u)->transfer_flags = 0;     }

#define FILL_DESC_OUT(state,ch,u,count) { int i; \
	unsigned shft = MFMT_BYTESSHIFT((state)->format); \
	unsigned len = count;  \
	for (i = 0 ; i < DESCFRAMES ; i++) { \
		(u)->iso_frame_desc[i].actual_length = 0; \
                (u)->iso_frame_desc[i].offset = 0; \
		(u)->iso_frame_desc[i].length = (len/(DESCFRAMES-i))&(~shft); \
		len -= (u)->iso_frame_desc[i].length; \
        } }

#define FILL_DESC_IN(state,ch,u,count) { int i, offs; \
        for ( i=0 , offs=0 ; i < DESCFRAMES; i++, offs += (ch)->maxsz) { \
	     (u)->iso_frame_desc[i].length = (ch)->maxsz; \
	     (u)->iso_frame_desc[i].offset = offs; } }

#define FILL_URB_OUT(state,ch,u,len) \
                 { FILL_URB(state,ch,u); FILL_DESC_OUT(state,ch,u,len);}
#define FILL_URB_IN(state,ch,u,len)  \
                 { FILL_URB(state,ch,u); FILL_DESC_IN(state,ch,u,len); }


/* --------------------------------------------------------------------- */


static int mi_init (struct st7554_modem_state *s)
{
	struct usb_modem_channel *ch = &s->mi;
	void *buf;
	int i, ret;

	ret = dma_init (&ch->dma);
	if (ret) return ret;

	/* urb init */
	buf = kmalloc( NUM_OF_URBS(ch)*ch->maxsz*DESCFRAMES, GFP_KERNEL);
	if (!buf)
		goto error;

	/* debug */
	memset(buf,0,NUM_OF_URBS(ch)*ch->maxsz*DESCFRAMES);

	for (i = 0 ; i < NUM_OF_URBS(ch) ; i++) {
		struct urb *u = usb_alloc_urb(DESCFRAMES);
		if (!u) {
			while(i)
				usb_free_urb(ch->urb[--i]);
			goto error1;
		}
		memset(u,0,sizeof(*u)+DESCFRAMES*sizeof(u->iso_frame_desc[0]));
		spin_lock_init(&u->lock);
		u->pipe = ch->pipe;
		u->transfer_buffer_length = ch->maxsz*DESCFRAMES;
		u->transfer_buffer = buf + i*(ch->maxsz*DESCFRAMES);
		FILL_URB_IN(s,ch,u,ch->maxsz*DESCFRAMES);
		ch->urb[i] = u;
	}

	return 0;
 error1:
	kfree(buf);
 error:
	dma_free(&ch->dma);
	return -ENOMEM;
}


static int mi_free(struct st7554_modem_state *s)
{
	struct usb_modem_channel *ch = &s->mi;
	void *buf = ch->urb[0]->transfer_buffer;
	int i;

	/* urb release */
	for (i = 0 ; i < NUM_OF_URBS(ch) ; i++)
		usb_free_urb(ch->urb[i]);

	kfree(buf);

	/* dma release */
	dma_free(&ch->dma);

	return 0;
}


static int mo_init (struct st7554_modem_state *s)
{
	struct usb_modem_channel *ch = &s->mo;
	int i, ret;

	ret = dma_init (&ch->dma);
	if (ret) return ret;

	/* urb init */
	for (i = 0 ; i < NUM_OF_URBS(ch) ; i++) {
		struct urb *u = usb_alloc_urb(DESCFRAMES);
		if (!u) {
			while(i)
				usb_free_urb(ch->urb[--i]);
			goto error;
		}
		memset(u,0,sizeof(*u)+DESCFRAMES*sizeof(u->iso_frame_desc[0]));
		spin_lock_init(&u->lock);
		u->pipe = ch->pipe;
		u->transfer_buffer_length = ch->maxsz*DESCFRAMES;
		u->transfer_buffer = ch->dma.buf;
		FILL_URB_OUT(s,ch,u,ch->maxsz*DESCFRAMES);
		ch->urb[i] = u;
	}

	return 0;
 error:
	dma_free(&ch->dma);
	return -ENOMEM;
}


static int mo_free(struct st7554_modem_state *s)
{
	struct usb_modem_channel *ch = &s->mo;
	int i;

	/* urb release */
	for (i = 0 ; i < NUM_OF_URBS(ch) ; i++)
		usb_free_urb(ch->urb[i]);

	/* dma release */
	dma_free(&ch->dma);

	return 0;
}


/* ----------------------------------------------------------------------- */

static void do_modem_bh(void *data)
{
	struct st7554_modem_state *s = data;
	unsigned long flags;
	int cnt, frag, mapped;

	spin_lock_irqsave(&s->lock, flags);
	mapped = s->mapped;
	frag = s->fragsize;
	cnt  = s->mi.dma.count / frag;
	spin_unlock_irqrestore(&s->lock, flags);

	USB_DBG_URB("modem process start (%d): dmain %d, dmaout %d; delay %d\n",
		    cnt, s->mi.dma.count, s->mo.dma.count, s->delay);

	while(cnt--) {
		modem_process(s->modem);
		if (mapped) {
			spin_lock_irqsave(&s->lock, flags);
			s->mi.dma.count -= frag;
			s->mo.dma.count += frag;
			spin_unlock_irqrestore(&s->lock, flags);
		}
	}

	USB_DBG_URB("modem process finish: dmain %d, dmaout %d. delay %d.\n",
		    s->mi.dma.count, s->mo.dma.count, s->delay);
}


static void do_ring_bh(void *data)
{
	struct st7554_modem_state *s = data;
	if (s->modem)
		modem_ring(s->modem);
}


/* --------------------------------------------------------------------- */

static void st7554_interrupt(urb_t *urb)
{
	struct st7554_modem_state *s = urb->context;
	u32 *status = urb->transfer_buffer;
	u16 fifo_status;
	u16 gpio_status;

	if (urb->status) {
		if (urb->status != -ENOENT)  /* unlinked */
			USB_ERR("st7554 interrupt: bad status received: %d\n", 
				urb->status);
		return;
	}

	fifo_status = *status &0xffff;
	gpio_status = *status >> 16;
#if 0
	USB_DBG("interrupt: fifo %04x, gpio %04x...\n",
		fifo_status, gpio_status);
#endif
	/* fifo evaluation */
	if (fifo_status & SSI1_UNDERRUN ) {
		USB_ERR("st7554: fifo underrun!\n");
		if (s && s->modem)
			modem_error(s->modem);
	}
	if (fifo_status & SSI1_OVERRUN) {
		USB_ERR("st7554: fifo overrun!\n");
		if (s && s->modem)
			modem_error(s->modem);
	}

	/* gpio evaluation */
	if ((gpio_status & GPIO_RING1) && s) {
		queue_task(&s->ring_bh, &tq_immediate);
		mark_bh(IMMEDIATE_BH);
	}

	/* ignore HANDSET */
	/* 		if (*gpio_status & GPIO_HANDSET) */
	/* 			status |= IRQ_HANDSET; */
}

/* --------------------------------------------------------------------- */


static void mo_complete(struct urb *u)
{
	struct st7554_modem_state *s = u->context;
	struct dmabuf *db = &s->mo.dma;
	iso_packet_descriptor_t *p;
	unsigned long flags;
	int i;

	if (u->status) {
		if (u->status == -ENOENT)
			return; /* unlinked */
		USB_ERR("mo_complete %d: err: urb status %d.\n",
			MO_URB_NO(s,u), u->status);
	}

	spin_lock_irqsave(&s->lock, flags);
	for (i = 0 ; i < u->number_of_packets ; i++) {
		p = u->iso_frame_desc + i;
		if (p->status)
			USB_ERR("mo_complete %d: err: fr.%d status %d.\n",
				MO_URB_NO(s,u), i, p->status);

		s->delay -= p->actual_length;

		if (s->disbalance + (int)p->length > 0) {
			p->length += s->disbalance;
			s->disbalance = 0;
		}
		else {
			/* FIXME: striping may optimize case recovery,
			   but fully stripped urb will cause mem leak in
			   usb controller driver (usb-uhci.o) */
			s->disbalance += p->length - 2 ;
			p->length = 2;
		}

		if (p->length > s->mo.maxsz) {
			s->disbalance += p->length - s->mo.maxsz;
			p->length = s->mo.maxsz;
		}
		if (p->length > db->size - db->tail) {
			s->disbalance += p->length - (db->size - db->tail);
			p->length = db->size - db->tail;
		}
		p->offset = db->tail;
		db->tail = (db->tail + p->length)%db->size ;
		db->count -= p->length;
	}
	spin_unlock_irqrestore(&s->lock, flags);

	USB_DBG_URB("mo_complete %d: %d: sent %d.\n",
		    MO_URB_NO(s,u), u->start_frame, u->actual_length);
	u->start_frame += NUM_OF_URBS(&s->mo)*u->number_of_packets ;
}


static void mo_setup_complete(struct urb *u)
{
	struct st7554_modem_state *s = u->context;
	unsigned long flags;

	if (u->status) {
		if (u->status == -ENOENT)
			return; /* unlinked */
		USB_ERR("mo_setup_complete %d: err: urb status %d.\n",
			MO_URB_NO(s,u), u->status);
	}

#ifdef DEBUG_URB_PRINT
	USB_DBG_URB("mo_setup_complete %d: %d: sent %d.\n",
		    MO_URB_NO(s,u), u->start_frame, u->actual_length);
	{
		int i;
		for (i = 0 ; i < u->number_of_packets ; i++) {
			USB_DBG_URB("\t%d: len %d, offs %d.\n", i,
				    u->iso_frame_desc[i].length,
				    u->iso_frame_desc[i].offset);
		}
	}
#endif
	spin_lock_irqsave(&s->lock, flags);
	FILL_DESC_OUT(s,&s->mo,u,((s->srate)/(1000/DESCFRAMES))<<MFMT_BYTESSHIFT(s->format));
	spin_unlock_irqrestore(&s->lock, flags);
	u->start_frame += (u==s->mo.urb[0])? SKIPFRAMES : 0 ;
	u->complete = mo_complete;
	mo_complete(u);
}


/* ----------------------------------------------------------------------- */

static void mi_complete(urb_t *u)
{
	struct st7554_modem_state *s = u->context;
	iso_packet_descriptor_t *p;
	unsigned long flags;
	int i;

	if (u->status) {
		if (u->status == -ENOENT)
			return; /* unlinked */
		USB_ERR("mi_complete %d: err: urb status %d.\n",
			MI_URB_NO(s,u), u->status);
		u->status = 0;
		goto error;
	}

	spin_lock_irqsave(&s->lock, flags);
	for (i = 0 ; i < u->number_of_packets ; i++) {
		p = u->iso_frame_desc + i;
		if (p->status) {
			USB_ERR("mi_complete %d: err: fr.%d status %d.\n",
				MI_URB_NO(s,u), i, p->status);
		}

		s->delay += p->actual_length;
		dmabuf_copyin(&s->mi.dma, u->transfer_buffer + p->offset, p->actual_length);
		/* set length of out urb (in driven) */
		u->next->iso_frame_desc[i].length = p->actual_length;
	}

	/* schedule data processing, if need  */
	if(s->mi.dma.count/s->fragsize) {
		queue_task(&s->modem_bh, &tq_immediate);
		mark_bh(IMMEDIATE_BH);
	}
	spin_unlock_irqrestore(&s->lock, flags);
 error:
	USB_DBG_URB("mi_complete %d: %d: recv %d.\n",
		    MI_URB_NO(s,u), u->start_frame, u->actual_length);
	u->start_frame += NUM_OF_URBS(&s->mi)*u->number_of_packets ;
}


static void mi_setup_complete(urb_t *u)
{
	struct st7554_modem_state *s = u->context;
	iso_packet_descriptor_t *p;
	unsigned long flags;
	int i;

	if (u->status) {
		if (u->status == -ENOENT)
			return; /* unlinked */
		USB_ERR("mi_setup_complete %d: err: urb status %d.\n",
			MI_URB_NO(s,u), u->status);
		u->status = 0;
		goto error;
	}

	spin_lock_irqsave(&s->lock, flags);
	for (i = 0 ; i < u->number_of_packets ; i++) {
		p = u->iso_frame_desc + i;
		if (p->status) {
			USB_ERR("mi_setup_complete %d: err: fr.%d status %d.\n",
				MI_URB_NO(s,u), i, p->status);
		}

		s->delay += p->actual_length;
		dmabuf_copyin(&s->mi.dma, u->transfer_buffer + p->offset, p->actual_length);
	}

	/* update out disbalance if need */
	if (u->actual_length < BYTES_IN_FRAMES(s,(DESCFRAMES+SKIPFRAMES-1))) {
		s->disbalance -= BYTES_IN_FRAMES(s,(DESCFRAMES+SKIPFRAMES-1)) -
						 u->actual_length;
		USB_DBG("mi_setup_complete: update disbalance %d.\n",
			s->disbalance);
	}
	spin_unlock_irqrestore(&s->lock, flags);
 error:
#ifdef DEBUG_URB_PRINT
	USB_DBG_URB("mi_setup_complete %d: %d: recv %d.\n",
		    MI_URB_NO(s,u), u->start_frame, u->actual_length);
	for (i = 0 ; i < u->number_of_packets ; i++) {
		p = u->iso_frame_desc + i;
		USB_DBG_URB("\t%d: len %d, offs %d.\n",
			    i, p->actual_length, p->offset);
	}
#endif
	u->complete = mi_complete;
	u->start_frame += NUM_OF_URBS(&s->mi)*u->number_of_packets ;
}


/* --------------------------------------------------------------------- */

/*
 * Start process brief explanation:
 *
 *                    --------------- estimated start transfer
 *                   |
 *       |<--------------------->|
 * frame:| 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10| 11| 12|
 * -----------------------------------------------------------
 *       |           |     in urb 0      |     in urb 1      |
 *   in: |---|---|---|+++|+++|+++|+++|+++|+++|+++|+++|+++|+++|
 *       |     out urb 0     |           |     out urb 1     |
 *  out: |+++|+++|+++|+++|+++|---|---|---|+++|+++|+++|+++|+++|
 * -----------------------------------------------------------
 *       |<----DESCFRAMES--->|SKIPFRAMES |<----DESCFRAMES--->|
 *
 *  Lengths (in bytes per frame):
 *
 *  out = out0 + out1 = FIFO SIZE - 1 (to prevent fifo overflow) + in
 *   in =  in0 +  in1 = DESCFRAMES * 2 + SKIPFRAMES - 1 = 12 (estimated)
 *
 *   in0 = depends on real start point (determinated at in0 complete)
 *   in1 = DESCFRAMES+SKIPFRAMES-1
 *  out0 = DESCFRAMES+SKIPFRAMES+1 (allocated on start)
 *  out1 = FIFO SIZE - 1 + in0 + in1 - out0 =
 *         FIFO SIZE - 3 + in0  (disbalance fix determinated at in0 complete)
 *
 */


static int st7554_modem_start (void *dev)
{
	struct st7554_modem_state *s = dev;
	int outlen;
	int start_frame;
	int ret;

	USB_DBG ("st7554 modem start...\n");

	/* clear run params */
	s->delay = s->disbalance = 0;

	/* prepare urbs */
	FILL_URB_IN(s, &s->mi, s->mi.urb[0], s->mi.maxsz*DESCFRAMES);
	FILL_URB_IN(s, &s->mi, s->mi.urb[1], s->mi.maxsz*DESCFRAMES);
	outlen = BYTES_IN_FRAMES(s,DESCFRAMES+SKIPFRAMES+1);
	FILL_URB_OUT(s, &s->mo, s->mo.urb[0], outlen);
	outlen = ST7554_FIFO_SIZE + BYTES_IN_FRAMES(s,(12-1)) - outlen;
	FILL_URB_OUT(s, &s->mo, s->mo.urb[1], outlen);

	s->mi.urb[0]->complete = mi_setup_complete;
	s->mi.urb[1]->complete = mi_complete;
	s->mo.urb[0]->complete = s->mo.urb[1]->complete = mo_setup_complete;

	/* all urbs are composed as ring listed pairs:
	   mo0 <-> mi1 ; mi0 <-> mo */
	s->mi.urb[0]->next = s->mo.urb[1];
	s->mo.urb[1]->next = s->mi.urb[0];
	s->mi.urb[1]->next = s->mo.urb[0];
	s->mo.urb[0]->next = s->mi.urb[1];

	/* submit all urbs */
	start_frame = usb_get_current_frame_number(s->usbdev);

	do {
		start_frame += 1;
		s->mo.urb[0]->start_frame = start_frame ;
		ret = usb_submit_urb(s->mo.urb[0]);
	} while( ret == -EAGAIN);

	s->mi.urb[0]->start_frame = start_frame + SKIPFRAMES;
	s->mo.urb[1]->start_frame = start_frame + SKIPFRAMES + DESCFRAMES;
	s->mi.urb[1]->start_frame = start_frame + SKIPFRAMES + DESCFRAMES;

	if ( ret ||
	     (ret = usb_submit_urb(s->mi.urb[0])) ||
	     (ret = usb_submit_urb(s->mo.urb[1])) ||
	     (ret = usb_submit_urb(s->mi.urb[1])) ) {
		USB_ERR("st7554 modem start: cannot submit urb: %d. fr. %d\n",
			ret, start_frame);
		return ret;
	}
	/* debug */
	USB_DBG_URB("st7554 modem start: submitted urbs: "
		    "mo0(%p) %d, mi0(%p) %d, mo1(%p) %d, mi1(%p) %d.\n",
		    s->mo.urb[0], s->mo.urb[0]->start_frame, 
		    s->mi.urb[0], s->mi.urb[0]->start_frame,
		    s->mo.urb[1], s->mo.urb[1]->start_frame,
		    s->mi.urb[1], s->mi.urb[1]->start_frame	);

	/* start fifo */
	ret = s->set_reg(s, ST7554_SSI1_COUNTER, s->fragsize);
	if (ret < 0)
		return ret;
	/* set fifo mask */
	return s->set_reg(s, ST7554_FIFO_MASK, SSI1_UNDERRUN|SSI1_OVERRUN);
}


static int st7554_modem_stop (void *dev)
{
	struct st7554_modem_state *s = dev;

	USB_DBG ("st7554 modem stop...\n");

	/* clear fifo mask */
	s->set_reg(s, ST7554_FIFO_MASK, 0);

	/* stop channels */
	USB_DBG_URB("st7554_modem_stop: unlink mi  urbs...\n");
	usb_unlink_urb(s->mi.urb[0]);
	usb_unlink_urb(s->mi.urb[1]);
	USB_DBG_URB("st7554_modem_stop: unlink mo urbs...\n");
	usb_unlink_urb(s->mo.urb[0]);
	usb_unlink_urb(s->mo.urb[1]);
	USB_DBG_URB("st7554_modem_stop: in/out urbs unlinked.\n");

	/* stop fifo */
	s->set_reg(s, ST7554_SSI1_COUNTER, 0);

	/* flush buffers */
	s->mi.dma.count = s->mi.dma.head = s->mi.dma.tail = 0;
	s->mo.dma.count = s->mo.dma.head = s->mo.dma.tail = 0;

	USB_DBG ("st7554 modem stop: delay %d.\n", s->delay);

	return 0;
}


static int st7554_modem_write(void *dev, char *buffer, int count)
{
	struct st7554_modem_state *s = dev;
	unsigned long flags;
	int ret;
	USB_DBG ("st7554 modem write: %d...\n", count);
	if (s->mapped)
		return -ENXIO;;
	spin_lock_irqsave(&s->lock, flags);
	ret = dmabuf_copyin(&s->mo.dma, buffer, count);
	spin_unlock_irqrestore(&s->lock, flags);
	return ret;
}

static int st7554_modem_read(void *dev, char *buffer, int count)
{
	struct st7554_modem_state *s = dev;
	unsigned long flags;
	int ret;	
	USB_DBG ("st7554 modem read: %d...\n", count);
	if (s->mapped)
		return -ENXIO;;
	spin_lock_irqsave(&s->lock, flags);
	ret = dmabuf_copyout(&s->mi.dma, buffer, count);
	spin_unlock_irqrestore(&s->lock, flags);
	return ret;
}


static int st7554_modem_set_srate (void *dev, unsigned srate)
{
	struct st7554_modem_state *s = dev;
	unsigned long flags;
	if (s->srate == srate)
		return 0;
	if(s->codec.set_srate(s, srate))
		return -EINVAL;
	spin_lock_irqsave(&s->lock,flags);
	s->srate = srate;
	spin_unlock_irqrestore(&s->lock,flags);
	return 0;
}

static int st7554_modem_set_format (void *dev, unsigned format)
{
	struct st7554_modem_state *s = dev;
	unsigned long flags;
	if(format == MFMT_QUERY)
		return s->codec.set_format(s, MFMT_QUERY);
	if (s->format == format)
		return 0;
	if(s->codec.set_format(s, format))
		return -EINVAL;
	spin_lock_irqsave(&s->lock,flags);
	s->format = format;
	spin_unlock_irqrestore(&s->lock,flags);
	return 0;
}

static int st7554_modem_set_frag (void *dev, unsigned frag)
{
	struct st7554_modem_state *s = dev;
	unsigned long flags;
	spin_lock_irqsave(&s->lock,flags);
	s->fragsize = frag;
	spin_unlock_irqrestore(&s->lock,flags);
	return 0;
}

static int st7554_modem_set_hook(void *dev, unsigned hook)
{
	struct st7554_modem_state *s = dev;
	unsigned long flags;
	u16 val;
	if (s->hook == hook)
		return 0;
	val = s->gpio_out;
	if (hook == MODEM_HOOK_OFF)
		val |=  (GPIO_HOOK1_OFF|GPIO_LED_HOOK);
	else if (hook == MODEM_HOOK_ON)
		val &= ~(GPIO_HOOK1_OFF|GPIO_LED_HOOK);
	else
		return -EINVAL;
	if(s->set_reg(s, ST7554_GPIO_OUT, val))
		return -EIO;
	spin_lock_irqsave(&s->lock,flags);
	s->gpio_out = val;
	s->hook     = hook;
	spin_unlock_irqrestore(&s->lock,flags);
	return 0;
}

static int st7554_modem_ioctl(void *dev, unsigned int cmd, unsigned long arg)
{
	struct st7554_modem_state *s = dev;

	USB_DBG ("st7554 modem ioctl: cmd %x...\n", cmd);

	if (!s || !s->usbdev)
		return -ENODEV;

	switch (cmd) {
	case MDMCTL_CAPABILITIES:
		return -EINVAL;
	case MDMCTL_HOOKSTATE:
		return st7554_modem_set_hook(s, arg);
		/* OSS API */
	case MDMCTL_SPEED:
		return st7554_modem_set_srate(s, arg);
	case MDMCTL_GETFMTS:
		return st7554_modem_set_format(s, MFMT_QUERY);
	case MDMCTL_SETFMT:
		return st7554_modem_set_format(s, arg);
	case MDMCTL_SETFRAGMENT:
		return st7554_modem_set_frag(s, arg);
	case MDMCTL_MAPIOBUFS:
		{ buffmem_desc *buf = (buffmem_desc *)arg;
		if (!buf) return -EFAULT;
		buf->in_size   = s->mi.dma.size;
		buf->in_buffer = s->mi.dma.buf;
		buf->out_size  = s->mo.dma.size;
		buf->out_buffer= s->mo.dma.buf;
		s->mapped = 1;
		return 0;
		}
	case MDMCTL_UNMAPIOBUFS:
		s->mapped = 0;
		return 0;
	case MDMCTL_IODELAY:
		{ unsigned long flags; unsigned val;
		spin_lock_irqsave(&s->lock, flags);
		val = s->delay + 154 ;
		spin_unlock_irqrestore(&s->lock, flags);
		USB_DBG("st7554 modem ioctl: IODELAY = %d.\n", val);
		return val;
		}
	default:
		break;
	}
	return -ENOIOCTLCMD;
}


static void *st7554_modem_open(void *m, int minor)
{
	struct list_head *list;
	struct st7554_modem_state *s;
	int ret;

	USB_DBG("st7554 modem open...\n");

	down (&open_sem);
	list_for_each(list,&st7554_devs) {
		s = list_entry(list, struct st7554_modem_state, list);
		if (!s->modem)
			goto dev_found;
	}

	up(&open_sem);
	USB_ERR("st7554 modem open: no avaliable devices found.\n");
	return 0;

 dev_found:
	if (!s->usbdev) {
		up(&open_sem);
		USB_ERR("st7554 modem open: usbdev is empty.\n");
		return 0;
	}

	MOD_INC_USE_COUNT;

	s->modem = m;
	up(&open_sem);

	ret = mo_init(s);
	if ( ret < 0 ) {
		USB_ERR("st7554 modem: cannot init out channel.\n");
		goto error;
	}
	ret = mi_init (s);
	if ( ret < 0 ) {
		USB_ERR("st7554 modem: cannot init in  channel.\n");
		mo_free(s);
		goto error;
	}
	s->mapped = 0;

	/* init bottom halves */
	INIT_TQUEUE( &s->ring_bh, do_ring_bh, s);
	INIT_TQUEUE( &s->modem_bh, do_modem_bh, s);

	/* initialize device */
	if ( s->init && s->init(s) < 0 ) {
		USB_ERR("st7554 modem: cannot initialize device.\n");
		goto error;		
	}
	/* sample rate, format (default setting) */
	st7554_modem_set_hook  (s, MODEM_HOOK_ON);
	st7554_modem_set_srate (s, 9600);
	st7554_modem_set_format(s, MFMT_S16_LE);
	st7554_modem_set_frag  (s, 96);

	return s;
 error:
	s->modem = 0;
	MOD_DEC_USE_COUNT;
	return 0;
}


static void st7554_modem_release(struct st7554_modem_state *s)
{
	USB_DBG("st7554 modem release...\n");

	/* clear stop conditions */
	st7554_modem_stop(s);

	/* release device */
	if (s->release)
		s->release(s);
	s->mapped = 0;
	mo_free(s);
	mi_free(s);

	s->modem = 0;

        MOD_DEC_USE_COUNT;	
}


static int st7554_modem_close(void *dev)
{
	struct st7554_modem_state *s = dev;
	USB_DBG("st7554 modem close...\n");
	down(&open_sem);
	st7554_modem_release(s);
	up(&open_sem);
	return 0;
}

/* --------------------------------------------------------------------- */

static struct modem_driver st7554_modem_driver = {
	name: "usb st7554",
	start_minor: 2,
	num: 2,
	open:	st7554_modem_open,
	close:	st7554_modem_close,
	start:  st7554_modem_start,
	stop:   st7554_modem_stop,
	ioctl:	st7554_modem_ioctl,
	read:   st7554_modem_read,
	write:  st7554_modem_write,
};

/* --------------------------------------------------------------------- */

#if 0
static int usb_modem_get_class_reg (struct st7554_modem_state *s,
				     u8 reg, void *data, int len)
{
	return usb_control_msg(s->usbdev,
			       usb_rcvctrlpipe(s->usbdev,s->ctrl_ep),
			       reg ,
			       USB_TYPE_CLASS|USB_RECIP_ENDPOINT,
			       0, s->ctrl_ep, data, len, HZ);
}

static int usb_modem_set_class_reg(struct st7554_modem_state *s,
				    u8 reg, u16 value)
{
	return usb_control_msg(s->usbdev,
			       usb_sndctrlpipe(s->usbdev,s->ctrl_ep),
			       reg ,
			       USB_TYPE_CLASS|USB_RECIP_ENDPOINT,  
			       value, s->ctrl_ep, NULL, 0, HZ);
}
#endif


/* ---------------------------------------------------------------------- */


static int st7554_get_reg (struct st7554_modem_state *s, u8 reg, u16 *val)
{
	int ret;
	ret = usb_control_msg(s->usbdev,
			      usb_rcvctrlpipe(s->usbdev,s->ctrl_ep),
			      reg|USB_DIR_IN,
			      USB_TYPE_VENDOR|USB_RECIP_DEVICE|USB_DIR_IN,  
			      0, 0, val, sizeof(*val),
			      CONTROL_MSG_TMO);
	if ( ret < 0 )
		USB_ERR("st7554_get_reg: error: reg %x, ret = %d\n",
		    reg, ret);
	return ret;
}


static int st7554_set_reg(struct st7554_modem_state *s, u8 reg, u16 value)
{
	int ret =
		usb_control_msg(s->usbdev,
				usb_sndctrlpipe(s->usbdev,s->ctrl_ep),
				reg,
				USB_TYPE_VENDOR|USB_RECIP_DEVICE,
				value, 0, NULL, 0,
				CONTROL_MSG_TMO);
	if (ret < 0 )
		USB_ERR("st7554_set_reg: error: reg %x, val %x, ret = %d\n",
			reg, value, ret);
	return ret;
}

static int stlc7550_set_srate(struct st7554_modem_state *s, unsigned int srate)
{
	if ( srate == 8000 )
		return s->set_reg(s, ST7554_SSI1_CWORD,0x3c8);
	else if (srate == 9600 )
		return s->set_reg(s, ST7554_SSI1_CWORD,0x3c0);
	else if (srate == 16000 )
		return s->set_reg(s, ST7554_SSI1_CWORD,0x3f0);
	else
		return -EINVAL;
}

static int stlc7550_set_format(struct st7554_modem_state *s, unsigned int format)
{
	if (format == MFMT_QUERY)
		return MFMT_U16_LE | MFMT_S16_LE;
	if (!MFMT_IS_16BIT(format))
		return -EINVAL;
	if (!MFMT_IS_LE(format))
		return -EINVAL;
	return 0;
}

#if 0
static int st75951_set_srate(struct st7554_modem_state *s, unsigned int srate)
{
	return -EINVAL;
}

static int st75951_set_format(struct st7554_modem_state *s, unsigned int format){
	return -EINVAL;
}
#endif

/* ---------------------------------------------------------------------- */

#if 0 /* ifdef DEBUG */
#define PRINT_REG(s ,reg, name) { u16 val; int ret; \
                   ret = s->get_reg(s,reg,&val);\
                   USB_DBG("st7554: vendor reg %s (%x) = %x , ret %d.\n", \
                   name, reg, val, ret); }

static void print_all_regs(struct st7554_modem_state *s) {
	PRINT_REG(s, ST7554_REVISION,"REVISION");
	PRINT_REG(s, ST7554_SSI1_CONTROL,"SSI1_CONTROL");
	PRINT_REG(s, ST7554_SSI2_CONTROL,"SSI2_CONTROL");
	PRINT_REG(s, ST7554_FIFO_MASK,"FIFO_MASK");
	PRINT_REG(s, ST7554_FIFO_SSI1_COUNTER,"FIFO_SSI1_COUNTER");
	PRINT_REG(s, ST7554_FIFO_SSI2_COUNTER,"FIFO_SSI2_COUNTER");
	PRINT_REG(s, ST7554_GPIO_DIR,"GPIO_DIR");
	PRINT_REG(s, ST7554_GPIO_OUT,"GPIO_OUT");
	PRINT_REG(s, ST7554_GPIO_MASK,"GPIO_MASK");
	PRINT_REG(s, ST7554_GPIO_INV,"GPIO_INV");
	PRINT_REG(s, ST7554_GPIO_STATUS,"GPIO_STATUS");
	PRINT_REG(s, ST7554_FIFO_CONTROL1,"FIFO_CONTROL1");
	PRINT_REG(s, ST7554_FIFO_CONTROL2,"FIFO_CONTROL2");
}
#endif /* DEBUG */

/* ---------------------------------------------------------------------- */

#define SET_REG(s,reg,val) { ret = s->set_reg(s,reg,val); if (ret < 0) { USB_ERR("st7554: failed to set reg %x.\n", reg); ; return ret;} }

#define CLEAR_ENDPOINT(s,pipe) { \
     if (usb_endpoint_halted(s->usbdev, usb_pipeendpoint(pipe), usb_pipeout(pipe))) {   \
       USB_DBG("st7554_init: pipe %d is halted. clear...\n", usb_pipeendpoint(pipe));  \
       if (!(ret=usb_clear_halt(s->usbdev, pipe))) return ret;}}



static int st7554_init  (struct st7554_modem_state *s)
{
	int ret;
	s->gpio_out = 0;
	SET_REG(s, ST7554_GPIO_DIR, 0xdff8);
	SET_REG(s, ST7554_GPIO_OUT, 0x00);
	/* SET_REG(s, ST7554_GPIO_MASK, GPIO_HANDSET); */
	SET_REG(s, ST7554_GPIO_MASK, GPIO_RING1);
	SET_REG(s, ST7554_FIFO_CONTROL1, 0x2828|BYTE_ORDER_LE);
	SET_REG(s, ST7554_FIFO_CONTROL2, 0x2828);
	SET_REG(s, ST7554_FIFO_MASK, 0x00);
	SET_REG(s, ST7554_FIFO_MASK, SSI1_UNDERRUN|SSI1_OVERRUN);
	SET_REG(s, ST7554_SSI1_COUNTER, 0x00);
	SET_REG(s, ST7554_SSI2_COUNTER, 0x00);
	/* power up */
	SET_REG(s, ST7554_SSI1_CONTROL, SSI1_POWERUP);
	/* control word */
	SET_REG(s, ST7554_SSI1_CWORD, 0x3c0);
	/* no inversion */
	SET_REG(s, ST7554_GPIO_INV, 0);

	/* clear usb ep */
	CLEAR_ENDPOINT(s, s->mi.pipe);
	CLEAR_ENDPOINT(s, s->mo.pipe);

	/* submit interrupt request */
	s->intr_urb.dev = s->usbdev;
	ret = usb_submit_urb(&s->intr_urb);
	if (ret < 0) {
		USB_ERR("st7554_init: cannot submit interrupt urb: %d.\n",
			ret);
		return ret;
	}

	return 0;
}

static int st7554_release (struct st7554_modem_state *s)
{
	int ret;
	/* unlink interrupt urb */
	usb_unlink_urb(&s->intr_urb);
	/* clear fifo & gpio  masks */
	SET_REG(s, ST7554_FIFO_MASK, 0);
	SET_REG(s, ST7554_GPIO_MASK, 0);
	/* hook on && all */
	s->gpio_out = 0;
	SET_REG(s, ST7554_GPIO_OUT, 0x00);
	/* power down */
	SET_REG(s, ST7554_SSI1_CONTROL, SSI1_POWERDOWN);
	return 0;
} 

/* --------------------------------------------------------------------- */

static void *st7554_probe(struct usb_device *usbdev, unsigned int ifnum,
			  const struct usb_device_id *id);
static void st7554_disconnect(struct usb_device *usbdev, void *ptr);

static struct usb_device_id st7554_ids [] = {
	{ USB_DEVICE(ST7554_ID_VENDOR,ST7554_ID_PRODUCT) },
	{ 0 }			   /* Terminating entry */
};

MODULE_DEVICE_TABLE (usb, st7554_ids);

static struct usb_driver st7554_usb_driver = {
	name:		"ST7554 USB Modem",
	probe:		st7554_probe,
	disconnect:	st7554_disconnect,
	driver_list:	LIST_HEAD_INIT(st7554_usb_driver.driver_list), 
	id_table:	st7554_ids,
};

/* --------------------------------------------------------------------- */


static void *st7554_probe(struct usb_device *usbdev, unsigned int ifnum,
			      const struct usb_device_id *id) {
	struct usb_interface *iface;
	struct usb_interface_descriptor *ifdesc;
	struct st7554_modem_state *s;
	unsigned int intr_ep;
	u16 val;

	if ( ifnum != 0 )
		return NULL;
	iface = usbdev->actconfig->interface;
	if ( iface->act_altsetting != ST7554_IFACE_MODEM_ALTSETTING )
		return NULL;

	USB_INFO("st7554 usb: probe if %02x, alt %02x...\n",
		 ifnum,iface->act_altsetting);

	ifdesc = iface->altsetting + iface->act_altsetting;
	if (usb_interface_claimed(iface)) {
		USB_ERR("st7554 probe: interface is busy.\n");
		return NULL;
	}

	s = kmalloc(sizeof(*s), GFP_KERNEL);
	if (!s) {
		USB_ERR("st7554 probe: no memory.\n");
		return NULL;
	}
	memset(s, 0, sizeof(*s));

	s->name = "ST7554 USB Modem";

	s->usbdev = usbdev;
	s->iface  = iface;

	s->ctrl_ep = 0;

	/* interrupt init */
	intr_ep  = ifdesc->endpoint[0].bEndpointAddress & 0x0f;
	FILL_INT_URB(&s->intr_urb, s->usbdev,
		     usb_rcvintpipe(s->usbdev,intr_ep),
		     &s->intr_status,
		     sizeof(s->intr_status),
		     st7554_interrupt, s,
		     ifdesc->endpoint[0].bInterval);
	s->intr_urb.transfer_flags = 0;

	s->mo.pipe  = usb_sndisocpipe(usbdev, 2);
	s->mo.maxsz = ifdesc->endpoint[1].wMaxPacketSize;
	s->mi.pipe  = usb_rcvisocpipe(usbdev, 3);
	s->mi.maxsz = ifdesc->endpoint[2].wMaxPacketSize;

	s->get_reg = st7554_get_reg;
	s->set_reg = st7554_set_reg;

	s->init       = st7554_init;
	s->release    = st7554_release;

	/* SSI1 codec type detection */
	if (s->get_reg(s,ST7554_GPIO_STATUS,&val) < 0) {
		USB_ERR("st7554 probe: cannot detect codec type.\n");
		goto error;
	}

	if (val&GPIO_MONOLITHINC) { /* st75951/2 silicon DAA codec */
		USB_ERR("st7554 probe: unsupported codec st75951/2.\n");
		s->codec.name = "stlc75971/2";
		goto error;
	}
	else {
		USB_DBG("codec stlc7550 detected.\n");
		s->codec.name = "stlc7550";
		s->codec.set_srate  = stlc7550_set_srate ;
		s->codec.set_format = stlc7550_set_format;
	}

	/* done by open/close */
	//if(st7554_init(s)) {
	//	USB_ERR("st7554 probe: cannot initialize device.\n");
	//	goto error;
	//}

	usb_set_configuration(usbdev, usbdev->actconfig->bConfigurationValue);
	usb_set_interface(usbdev, 0 , ST7554_IFACE_MODEM_ALTSETTING);
	usb_driver_claim_interface(&st7554_usb_driver, s->iface, s);

	down(&open_sem);
	if (list_empty(&st7554_devs)
	    && register_modem_driver(&st7554_modem_driver)) {
		USB_ERR("st7554_probe: cannot register modem driver.\n");
	}
	list_add_tail(&s->list, &st7554_devs);
	up(&open_sem);

	return s;
 error:
	kfree(s);
	return NULL;
}


static void st7554_disconnect(struct usb_device *usbdev, void *ptr) {
	struct st7554_modem_state *s = ptr;
	USB_DBG("st7554 disconnect...\n");
        if (!s) {
                USB_DBG("st7554 disconnect: no dev.\n");
                return;
        }

        if (!s->usbdev) {
                USB_DBG("st7554 disconnect: already called for %p!\n", s);
                return;
        }

	down(&open_sem);
	list_del(&s->list);
	/* notify closing */
	if (s->modem) {
		modem_release(s->modem);
		st7554_modem_release(s);
	}
	/* done by open/close */
	//st7554_release(s);

	s->usbdev = NULL;
	if (list_empty(&st7554_devs))
		unregister_modem_driver(&st7554_modem_driver);
	up(&open_sem);

	usb_driver_release_interface(&st7554_usb_driver, s->iface);
	kfree(s);
}

/* ---------------------------------------------------------------------- */


static int __init st7554_modem_init(void)
{
	int ret;
	USB_INFO ("ST7554 USB Modem.\n");
	ret = usb_register(&st7554_usb_driver);
	if ( ret ) {
		USB_ERR ("st7554_modem_init: cannot register usb device.\n");
		return ret;
	}
	return 0;
}


static void __exit st7554_modem_exit(void)
{
	USB_DBG ("st7554: exit...\n");
	usb_deregister(&st7554_usb_driver);
}


module_init(st7554_modem_init);
module_exit(st7554_modem_exit);

EXPORT_NO_SYMBOLS;

MODULE_AUTHOR("Smart Link Ltd.");
MODULE_DESCRIPTION("ST7554 USB Smart Link Soft Modem driver.");
MODULE_LICENSE("Smart Link Ltd.");

