/*****************************************************************************/

/*
 *
 *   Copyright (c) 2002, Smart Link Ltd.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *       1. Redistributions of source code must retain the above copyright
 *          notice, this list of conditions and the following disclaimer.
 *       2. Redistributions in binary form must reproduce the above
 *          copyright notice, this list of conditions and the following
 *          disclaimer in the documentation and/or other materials provided
 *          with the distribution.
 *       3. Neither the name of the Smart Link Ltd. nor the names of its
 *          contributors may be used to endorse or promote products derived
 *          from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *
 *	amrmo_init.c  -  Smart Link Soft Modem amr,pci driver initialization.
 *
 *	Author: Sasha K (sashak@smlink.com)
 *
 */

/*****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/kernel.h>

#include <linux_compat.h>

#define FORCE_VENDOR_DEVICE 1
#undef CONFIG_PM

#if MODEM_DEBUG
#include <modem_debug.h>
#undef printk
#define printk dprintf
#define AMRMO_DBG(fmt...) printk(KERN_DEBUG fmt)
#else
#define AMRMO_DBG(fmt...)
#endif

/* modem cards pci ids list */

/* SmartRisers */

/* Intel */
#define PCI_DEVICE_ID_ICHAA			0x2416
#define PCI_DEVICE_ID_ICHAB			0x2426
#define PCI_DEVICE_ID_BANISTER			0x7196
#define PCI_DEVICE_ID_ICH2			0x2446
#define PCI_DEVICE_ID_ICH3			0x2486
#define PCI_DEVICE_ID_ICH4			0x24C6
/* Standard Microsystems */
#define PCI_VENDOR_ID_STANDARD_MICROSYSTEM	0x1055	
#define PCI_DEVICE_ID_STANDARD_MICROSYSTEM	0x9178
/* AMD */
#define PCI_DEVICE_ID_AMD_ACLINK		0x7446
/* NVidia */
#define PCI_DEVICE_ID_MCP			0x01C1
/* VIA */
#define PCI_DEVICE_ID_VIA			0x3068
/* SiS */
#define PCI_DEVICE_ID_SIS630			0x7013
#define PCI_DEVICE_ID_SIS960			0x7018
/* ALi */
#define PCI_DEVICE_ID_ALI5450			0x5450
#define PCI_DEVICE_ID_ALI5451			0x5451
#define PCI_DEVICE_ID_ALI5457			0x5457


/* SmartPCIs */		

#define PCI_VENDOR_ID_SMARTLINK 		0x163C
#define PCI_VENDOR_ID_SMARTLINK_1		0x163C
#define PCI_VENDOR_ID_SMARTLINK_2		0x10A5		

#define PCI_DEVICE_ID_SL1500			0x3400
#define PCI_DEVICE_ID_SL1800			0x5459
#define PCI_DEVICE_ID_SL1801			0x545A
#define PCI_DEVICE_ID_SL1900			0x3052


#define PCI_DEVICE_ID_ALS300p			0x0308
#define PCI_DEVICE_ID_SMARTLAN56		0x8197


enum {
	ALS300_CARD = 1,
	VIA3058_CARD,
	ALI5450_CARD,
	ALI5451_CARD,
	SL1800_CARD,
	SIS630_CARD,
	SIS960_CARD,
	ICH_CARD,
	ICH4_CARD,
	SMARTLAN56_CARD,
	NVIDIA_CARD,
	SL1500_CARD,
	SL1801_CARD,
	SL1900_CARD
};


struct amr_card {
	unsigned id;
	const char *name;
	struct pci_dev *pci_dev;
	unsigned int irq;
	unsigned long iobase1;
	unsigned long iobase2;
	unsigned long membase;
	unsigned long memlen;
	void *data;
};


extern void *create_amrmo_card(int card_id, void *card);
extern void delete_amrmo_card(void *data);
extern void interrupt_amrmo_card(void *data);

#ifdef FORCE_VENDOR_DEVICE
static unsigned short vendor = 0;
static unsigned short device = 0;
MODULE_PARM(vendor,"h");
MODULE_PARM_DESC(vendor, "Vendor ID");
MODULE_PARM(device,"h");
MODULE_PARM_DESC(device, "Device ID");
#endif

static const char *card_names[] = {
	0,
	"ALS300+",
	"VIA3058/SmartRiser",
	"ALI1535/SmartRiser",
	"ALI1535/SmartRiser",
	"SL1800(SmartPCI561)",
	"SiS630/SmartRiser",
	"SiS960/SmartRiser",
	"ICH/SmartRiser",
	"ICH4/SmartRiser",
	"SmartLAN56",
	"NvidiaMCP/SmartRiser",
	"SL1500(SmartPCI56)",
	"SL1801(SmartPCI563)",
	"SL1900(SmartPCI562)"
};


static struct pci_device_id amrmo_pci_tbl [] __devinitdata = {
	{PCI_VENDOR_ID_SMARTLINK_1, PCI_DEVICE_ID_SL1900,  /* 163c:3052 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SL1900_CARD},
	{PCI_VENDOR_ID_SMARTLINK_2, PCI_DEVICE_ID_SL1900,  /* 10a5:3052 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SL1900_CARD},
	{PCI_VENDOR_ID_PHILIPS,  PCI_DEVICE_ID_SL1500,     /* 1131:3400 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SL1500_CARD},
	{PCI_VENDOR_ID_REALTEK,  PCI_DEVICE_ID_SMARTLAN56, /* 10ec:8197 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SMARTLAN56_CARD},
	{PCI_VENDOR_ID_AL, PCI_DEVICE_ID_SL1800,           /* 10b9:5459 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SL1800_CARD},
	{PCI_VENDOR_ID_SMARTLINK_1, PCI_DEVICE_ID_SL1800,  /* 163c:5459 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SL1800_CARD},
	{PCI_VENDOR_ID_SMARTLINK_2, PCI_DEVICE_ID_SL1800,  /* 10a5:5459 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SL1800_CARD},
	{PCI_VENDOR_ID_AVANCE, PCI_DEVICE_ID_ALS300p,      /* 4005:0308 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ALS300_CARD},
	{PCI_VENDOR_ID_STANDARD_MICROSYSTEM,
		PCI_DEVICE_ID_STANDARD_MICROSYSTEM,        /* 1055:9178 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ICH_CARD},
	{PCI_VENDOR_ID_AMD, PCI_DEVICE_ID_AMD_ACLINK,      /* 1022:7446 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ICH_CARD},
	{PCI_VENDOR_ID_NVIDIA, PCI_DEVICE_ID_MCP,          /* 10de:01c1 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, NVIDIA_CARD},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_ICHAA,         /* 8086:2416 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ICH_CARD},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_ICHAB,         /* 8086:2426 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ICH_CARD},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_BANISTER,      /* 8086:7196 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ICH_CARD},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_ICH2,          /* 8086:2446 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ICH_CARD},
	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_ICH3,          /* 8086:2486 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ICH_CARD},
       	{PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_ICH4,
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, ICH4_CARD},
	{PCI_VENDOR_ID_VIA, PCI_DEVICE_ID_VIA,             /* 1106:3068 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, VIA3058_CARD},
	{PCI_VENDOR_ID_SI, PCI_DEVICE_ID_SIS630,           /* 1039:7013 */
	 PCI_ANY_ID, PCI_ANY_ID, PCI_CLASS_COMMUNICATION_MODEM<<8, 0xff00, SIS630_CARD},
	{PCI_VENDOR_ID_SI, PCI_DEVICE_ID_SIS960,           /* 1039:7018 */
	 PCI_ANY_ID, PCI_ANY_ID, PCI_CLASS_COMMUNICATION_MODEM<<8, 0xff00, SIS960_CARD},
	//{PCI_VENDOR_ID_AL, PCI_DEVICE_ID_ALI5450,
	// PCI_ANY_ID, PCI_ANY_ID, 0, 0, ALI5450_CARD},
	//{PCI_VENDOR_ID_AL, PCI_DEVICE_ID_ALI5451,
	// PCI_ANY_ID, PCI_ANY_ID, 0, 0, ALI5451_CARD},
	{PCI_VENDOR_ID_AL, PCI_DEVICE_ID_ALI5457,          /* 10b9:5457 */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SL1800_CARD},
	{PCI_VENDOR_ID_AL, PCI_DEVICE_ID_SL1801,           /* 10b9:545a */
	 PCI_ANY_ID, PCI_ANY_ID, 0, 0, SL1801_CARD},
	{0,}
};

MODULE_DEVICE_TABLE (pci, amrmo_pci_tbl);


void amr_mod_usage_increase() {
	MOD_INC_USE_COUNT;
}

void amr_mod_usage_decrease() {
	MOD_DEC_USE_COUNT;
}

static void amrmo_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
        struct amr_card *card = (struct amr_card *)dev_id;
	interrupt_amrmo_card(card->data);
}

#ifdef CONFIG_PM
static int amrmo_suspend(struct pci_dev *dev, u32 unused)
{
	AMRMO_DBG("amrmo_suspend...\n");
	return 0;
}

static int amrmo_resume(struct pci_dev *dev)
{
	AMRMO_DBG("amrmo_resume...\n");
	return 0;
}
#endif /* CONFIG_PM */


static int __init amrmo_probe(struct pci_dev *pci_dev, const struct pci_device_id *pci_id)
{
	struct amr_card *card;
	printk(KERN_INFO "amrmo: probe %04x:%04x %s : %s card...\n",
	       pci_id->vendor,pci_id->device,pci_dev->name,
	       card_names[pci_id->driver_data]);
#ifdef FORCE_VENDOR_DEVICE
	if ( (vendor && pci_id->vendor != vendor) ||
	     (device && pci_id->device != device) ) {
		printk(KERN_WARNING
		       "force configured for only %04x/%04x cards.\n",
		       vendor, device);
		return -ENODEV;
	}
#endif
	if ( pci_enable_device(pci_dev) ||
	     pci_dev->irq == 0 )
		return -ENODEV;

	card = kmalloc(sizeof(*card), GFP_KERNEL);
        if (!card)
                return -ENOMEM;
        memset(card, 0, sizeof(*card));

	card->id = pci_id->driver_data;
	card->name = card_names[card->id];
	card->pci_dev = pci_dev;
	card->irq     = pci_dev->irq;
	if (pci_resource_flags(pci_dev,0)&IORESOURCE_MEM) {
		card->membase = pci_resource_start(pci_dev, 0);
		card->iobase1 = pci_resource_start(pci_dev, 1);
	}
	else if (pci_resource_flags(pci_dev,1)&IORESOURCE_MEM) {
		card->iobase1 = pci_resource_start(pci_dev, 0);
		card->membase = pci_resource_start(pci_dev, 1);
	}
	else {
		card->iobase1 = pci_resource_start(pci_dev, 0);
		card->iobase2 = pci_resource_start(pci_dev, 1);
	}

	card->data = create_amrmo_card(pci_id->driver_data,card);
	AMRMO_DBG("amrmo_probe: card is %p. data %p, pci %p, io %lx, %lx.\n",
		  card, card->data, card->pci_dev, card->iobase1,card->iobase2);
	if (!card->data) {
		kfree(card);
		return -ENOMEM;
	}

	pci_set_master(pci_dev);
        /* claim iospaces and irq */
	if(pci_request_regions(pci_dev,(char*)card->name)) {
		printk(KERN_ERR "amrmo: failed request regions.\n");
		goto error_out;
	}
        if(request_irq(card->irq, &amrmo_interrupt,SA_SHIRQ,card->name,card)) {
		printk(KERN_ERR "amrmo: failed request_irq\n");
		pci_release_regions(pci_dev);
		goto error_out;
	}

	pci_set_drvdata(pci_dev, card);
	return 0;

 error_out:
	delete_amrmo_card(card->data);
	kfree(card);
	pci_disable_device(pci_dev);
	return -ENODEV;
}

static void __exit amrmo_remove(struct pci_dev *pci_dev)
{
	struct amr_card *card = pci_get_drvdata(pci_dev);
	AMRMO_DBG("amrmo: remove %p...\n", card);
        free_irq(card->irq, card);
	pci_release_regions(pci_dev);
	delete_amrmo_card(card->data);
	pci_set_drvdata(pci_dev, NULL);
	kfree(card);
	pci_disable_device(pci_dev);
	return;
}

/*
 *  module stuff
 */

MODULE_AUTHOR("Smart Link Ltd.");
MODULE_DESCRIPTION("Smart Link HAMR5600,SmartPCI56/561 based modem driver");
MODULE_LICENSE("Smart Link Ltd.");


#define AMRMO_MODULE_NAME "slamrmo"

static struct pci_driver amrmo_pci_driver = {
	name:		AMRMO_MODULE_NAME,
	id_table:	amrmo_pci_tbl,
	probe:		amrmo_probe,
	remove:		amrmo_remove,
#ifdef CONFIG_PM
	suspend:	amrmo_suspend,
	resume:		amrmo_resume
#endif /* CONFIG_PM */
};

static int __init amrmo_init(void)
{
	struct pci_dev *dev;
	if (!pci_present())
		return -ENODEV;

	printk(KERN_INFO "Smart Link AMRMO modem.\n");

	/* fix me: how to prevent modem cards grabing by
	   serial driver? */
	pci_for_each_dev(dev) {
		if(pci_match_device(amrmo_pci_tbl, dev) &&
		   pci_dev_driver(dev)) {
			AMRMO_DBG("device %04x:%04x %s is used by %s: remove\n",
				  dev->vendor,dev->device, dev->name,
				  dev->driver?dev->driver->name:"");
			if (dev->driver && dev->driver->remove)
				dev->driver->remove(dev);
			dev->driver = NULL;
		}
	}

	if (!pci_register_driver(&amrmo_pci_driver)) {
		pci_unregister_driver(&amrmo_pci_driver);
                return -ENODEV;
	}
	return 0;
}

static void __exit amrmo_exit(void)
{
	AMRMO_DBG("amrmo: exit...\n");
	pci_unregister_driver(&amrmo_pci_driver);
}

module_init(amrmo_init);
module_exit(amrmo_exit);

EXPORT_NO_SYMBOLS;

